# Script for manuscript:
# "Heat-absorbing sexual coloration co-adapts with increased heat tolerance in dragonflies"

# load packages
library(phytools)
library(car)
library(MASS)
library(nlme)
library(ggplot2)
library(lme4)
library(lmerTest)
library(lsmeans)
library(ggplot2)
library(plyr)
library(dplyr)
library(letsR)
library(phylolm)
library(phylosignal)
library(geiger)
library(MuMIn)
library(phylobase)
library(phylopath)
library(evobiR)

#rstudio help package
library(rstudioapi)
library(Publish)

#set working directory to source location
setwd(dirname(getActiveDocumentContext()$path))  

#check the working directory
getwd()  



## load the dataframe and the phylogeny
tols <- read.csv('drags_ctmax_data_add.csv')
na.odes <- read.nexus('na_odes_add_phylogeny.nex')
plot(na.odes)

## name the rows of the dataframe after the species. 
rownames(tols) <- tols$binom

# to see which species in our dataframe are not in our phylogeny, we use the "match()" function. the second line of code calls up which species are missing
tols$matched <- match(tols$binom, na.odes$tip.label)
tols[which(is.na(tols$matched)), ]
# this returns 5 species that are in our data set but don't appear to be in our phylogeny
# so for these five species, we can fix three of them very easily. Erthemis is a mis-spelling, so we can correct that. Tramea walkeri has been renamed to Tramea binotata, so that's easy too. 
#The last one is Epigomphus. There's only one Epigomphid in the analysis, so even though that particular species isn't in the tree, the genus is represented, so we can arbitrarily choose one of the species that is represented in the genus and just rename the row for now. 
#Note that if there was more than one species in this genus in our analysis (like for the other two species), we could not do this. 
# rename the rows,  and check again
tols$binom <- revalue(tols$binom, c(
  'Epigomphus_quadracies' = 'Epigomphus_crepidus', 
  'Erthemis_simplicicollis' = 'Erythemis_simplicicollis',
  'Tramea_walkeri' = 'Tramea_binotata'))

rownames(tols) <- tols$binom
tols$matched <- match(tols$binom, na.odes$tip.label)

tols[which(is.na(tols$matched)),]
#                                 binom       Genus Species              label color  N Tmax lat_max matched
# Erythemis_credula   Erythemis_credula   Erythemis credula  Erythemis credula     n 14 45.6    9.16      NA
# Micrathyria_eximia Micrathyria_eximia Micrathyria  eximia Micrathyria eximia     n 14 44.7    9.16      NA

### these two species don't have known phylogenetic position. 

## make a new dataframe in R that has only the species with phylogenetic positions
tol.clean <- subset(tols, !is.na(matched))



#### PGLS ####
# do an AIC comparison of a brownian motion model, a model that estimates Pagel's lambda, and a model that doesn't include phylogeny.

tol00.brownian <- gls(Tmax ~ color, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'REML', na.action = na.fail)
tol00.pagels <- gls(Tmax ~ color, correlation = corPagel(1, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.pagels.5 <- gls(Tmax ~ color, correlation = corPagel(0.5, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.pagels.0 <- gls(Tmax ~ color, correlation = corPagel(0, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.star <- gls(Tmax ~ color, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'REML', na.action =  na.fail)

# now let's see which of those models is the best fit to the data using AIC
model.sel(tol00.brownian, tol00.pagels, tol00.pagels.5, tol00.pagels.0, tol00.star, rank = "AIC")
#                (Intrc) color   correlation df  logLik  AIC delta weight
# tol00.star       43.88     + cP(0,na.od,T)  3 -37.251 80.5  0.00  0.339
# tol00.pagels     44.29     +   cP(1,na.od)  4 -36.684 81.4  0.86  0.220
# tol00.pagels.0   44.29     +   cP(0,na.od)  4 -36.684 81.4  0.86  0.220
# tol00.pagels.5   44.29     + cP(0.5,na.od)  4 -36.684 81.4  0.86  0.220
# tol00.brownian   44.71     + cP(1,na.od,T)  3 -42.877 91.8 11.25  0.001
# Pagel's converges

model.sel(tol00.brownian, tol00.pagels, tol00.star, rank = "AIC")
#                (Intrc) color   correlation df  logLik  AIC delta weight
# tol00.star       43.88     + cP(0,na.od,T)  3 -37.251 80.5  0.00  0.605
# tol00.pagels     44.29     +   cP(1,na.od)  4 -36.684 81.4  0.86  0.393
# tol00.brownian   44.71     + cP(1,na.od,T)  3 -42.877 91.8 11.25  0.002

# The star and Pagel's lambda model have the lowest AIC. Lets report results for all three to cover a range of phylogenetic signal values

# # Likelihood ratio tests pagels
# problems with model if starting lambda value isn't specified, so starting at 0.5
tol01a <- gls(Tmax ~ color, correlation = corPagel(0.5, phy = na.odes), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b <- gls(Tmax ~ 1, correlation = corPagel(0.5, phy = na.odes), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color on tmax
anova(tol01a, tol01b)
#       Model df      AIC      BIC    logLik   Test L.Ratio p-value
# tol01a     1  4 67.10906 70.88682 -29.55453                       
# tol01b     2  3 78.36176 81.19508 -36.18088 1 vs 2 13.2527   3e-04

# estimated lambda is negative, so could be that closely related species are extremely divergent in ctmax. probably best to just focus on star phylogeny results
summary(tol01a) #0.88644 estimate
confint(tol01a) #-1.020023  2.792912 CI


# Likelihood ratio tests brownian
tol01a.brownian <- gls(Tmax ~ color, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b.brownian <- gls(Tmax ~ 1, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color on tmax
anova(tol01a.brownian, tol01b.brownian)
#                Model df      AIC      BIC    logLik   Test L.Ratio p-value
# tol01a.brownian     1  3  98.0415 100.8748 -46.02075                       
# tol01b.brownian     2  2 109.1715 111.0604 -52.58575 1 vs 2   13.13   3e-04

summary(tol01a.brownian) #2.96436 estimate
confint(tol01a.brownian) #1.552263  4.376464 CI


# Likelihood ratio tests star
tol01a.star <- gls(Tmax ~ color, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b.star <- gls(Tmax ~ 1, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color on tmax
anova(tol01a.star, tol01b.star)
#            Model df      AIC      BIC    logLik   Test  L.Ratio p-value
# tol01a.star     1  3 82.15005 84.98337 -38.07503                        
# tol01b.star     2  2 85.54955 87.43843 -40.77478 1 vs 2 5.399499  0.0201

summary(tol01a.star) #2.08432 estimate
confint(tol01a.star) #0.3560839  3.812552 CI


tmax.estimates <- as.data.frame(emmeans(tol01a.star, "color"))






#### PGLS tropical species only ####

# subset dataframe to only include tropicer species
tol.clean.tropic <- filter(tol.clean, climate == "tropical")

# first we figure out which species are shared between the phylogeny and the subsetted dataframe
match_data_phylo.tropic <- match(na.odes$tip.label, tol.clean.tropic$binom)

# next we create a vector of names of species in the phylogeny that aren't in our dataframe
drop.species.tropic <- na.odes$tip.label[is.na(match_data_phylo.tropic)==TRUE]

# finally, we use the drop.tip() command to prune the tree so that includes only the species in our dataset
na.odes.tropic <- drop.tip(na.odes, drop.species.tropic)

# always a good idea to plot it
plot(na.odes.tropic)

# also a good idea to just look at the summary description of this phylogeny
summary(na.odes.tropic)


# now something that is kind of important for these kinds of analyses is that the phylogeny is "ultrametric", which essentially just means that the tips are all aligned. Sometimes you see phylogenies where the tips do not end in the same place, those are examples of phylogenies that are not ultrametric. From when we plotted the phylogeny, it sure looked like the tips were aligned, but it's always good to double check
is.ultrametric(na.odes.tropic) # TRUE

# do an AIC comparison of a brownian motion model, a model that estimates Pagel's lambda, and a model that doesn't include phylogeny at all. The way do to this is as such:
tol00.tropic.brownian <- gls(Tmax ~ color, correlation = corPagel(1, phy = na.odes.tropic, fixed = TRUE), data = tol.clean.tropic, method = 'REML', na.action = na.fail)
tol00.tropic.pagels <- gls(Tmax ~ color, correlation = corPagel(1, phy = na.odes.tropic), data = tol.clean.tropic, method = 'REML', na.action =  na.fail)
tol00.tropic.pagels.5 <- gls(Tmax ~ color, correlation = corPagel(0.4, phy = na.odes.tropic), data = tol.clean.tropic, method = 'REML', na.action =  na.fail)
tol00.tropic.pagels.0 <- gls(Tmax ~ color, correlation = corPagel(0, phy = na.odes.tropic), data = tol.clean.tropic, method = 'REML', na.action =  na.fail)
tol00.tropic.star <- gls(Tmax ~ color, correlation = corPagel(0, phy = na.odes.tropic, fixed = TRUE), data = tol.clean.tropic, method = 'REML', na.action =  na.fail)

# now let's see which of those models is the best fit to the data using AIC
model.sel(tol00.tropic.brownian, tol00.tropic.pagels, tol00.tropic.pagels.5, tol00.tropic.pagels.0, tol00.tropic.star, rank = "AIC")
#                  (Int) clr lat_bin   correlation df  logLik  AIC delta weight
# tol00.tropic.pagels.5   43.62     + c(0.4,n.o.t)  4 -25.760 59.5  0.00  0.306
# tol00.tropic.pagels     43.49     +   c(1,n.o.t)  4 -25.820 59.6  0.12  0.288
# tol00.tropic.star       43.22     + c(0,n.o.t,T)  3 -27.042 60.1  0.56  0.231
# tol00.tropic.pagels.0   43.72     +   c(0,n.o.t)  4 -26.320 60.6  1.12  0.175
# tol00.tropic.brownian   43.57     + c(1,n.o.t,T)  3 -32.841 71.7 12.16  0.001
# pagel's does not converge

model.sel(tol00.tropic.brownian, tol00.tropic.star, rank = "AIC")
#                       (Intrc) color  correlation df  logLik  AIC delta weight
# tol00.tropic.star       43.22     + c(0,n.o.t,T)  3 -27.042 60.1   0.0  0.997
# tol00.tropic.brownian   43.57     + c(1,n.o.t,T)  3 -32.841 71.7  11.6  0.003

# brownian
tol01.tropica.brownian <- gls(Tmax ~ color, correlation = corPagel(1, phy = na.odes.tropic, fixed = TRUE), data = tol.clean.tropic, method = 'ML', na.action =  na.fail )
tol01.tropicb.brownian <- gls(Tmax ~ 1, correlation = corPagel(1, phy = na.odes.tropic, fixed = TRUE), data = tol.clean.tropic, method = 'ML', na.action =  na.fail )
# test effect of color on tmax
anova(tol01.tropica.brownian, tol01.tropicb.brownian)
#                       Model df      AIC      BIC    logLik   Test  L.Ratio p-value
# tol01.tropica.brownian     1  3 78.11049 80.02767 -36.05525                        
# tol01.tropicb.brownian     2  2 88.68285 89.96097 -42.34143 1 vs 2 12.57236   4e-04

summary(tol01.tropica.brownian) #3.24938 estimate
confint(tol01.tropica.brownian) #1.725103  4.773651 CI

# star
tol01.tropica.star <- gls(Tmax ~ color, correlation = corPagel(0, phy = na.odes.tropic, fixed = TRUE), data = tol.clean.tropic, method = 'ML', na.action =  na.fail )
tol01.tropicb.star <- gls(Tmax ~ 1, correlation = corPagel(0, phy = na.odes.tropic, fixed = TRUE), data = tol.clean.tropic, method = 'ML', na.action =  na.fail )
# test effect of color on tmax
anova(tol01.tropica.star, tol01.tropicb.star)
#                   Model df      AIC      BIC    logLik   Test L.Ratio p-value
# tol01.tropica.star     1  3 62.42287 64.34004 -28.21144                       
# tol01.tropicb.star     2  2 66.41196 67.69008 -31.20598 1 vs 2 5.98909  0.0144

summary(tol01.tropica.star) # 2.68 estimate
confint(tol01.tropica.star) # 0.604725  4.755275 CI

tmax.estimates.tropic <- as.data.frame(emmeans(tol01.tropica.star, "color"))
tmax.estimates.tropic



#### code for data panels of Figure 1 ####

# specify theme

theme.drags <- theme(axis.text = element_text(color="Black"),
                     axis.title.x = element_text(size = 11), 
                     axis.text.x = element_text(size = 11),
                     axis.title.y = element_text(size = 11), 
                     axis.text.y = element_text(size = 9),
                     panel.border = element_blank(),
                     panel.background = element_blank(),
                     panel.grid.minor=element_blank(), panel.grid.major=element_blank(),
                     axis.line.x = element_line(linetype = 'solid', color = 'black', size = 0.5),
                     axis.line.y = element_line(linetype = 'solid', color = 'black', size = 0.5),
                     plot.margin = margin(1,1,5,1),
                     legend.key = element_rect(fill = 'white')) 

tmax_extimates_plot <-
  ggplot() +  
  #geom_hline(yintercept = 0, size = 0.5, color = "grey", linetype = "dashed") +
  geom_linerange(data = tmax.estimates, aes(x = color, ymax = upper.CL, ymin = lower.CL, color = color), size = 0.8, alpha = 1, ) + 
  geom_point(data = tmax.estimates, aes(x = color, y= emmean, color = color, fill = color), alpha = 1, size=3, stroke = 1, pch = 21) + 
  #scale_shape_manual(values = c(21, 16)) +
  #scale_x_discrete(labels = c("Temperate\nspecies", "Tropical\nspecies")) +
  scale_color_manual(values = c("#6F91A0", "#654C28")) +
  scale_fill_manual(values = c("#D9E8F3", "#EBBF68")) +
  #scale_y_continuous(breaks = c(12,15,18,21,24,27), limits = c(11,27)) +
  #scale_y_continuous(breaks = c(-0.4,-0.2,0,0.2,0.4)) +
  #ylab("Mean CTmax (ºC)") +
  theme.drags +                                 
  theme(legend.position = "none", axis.title.x = element_blank(), axis.title.y = element_blank(), axis.text.x = element_blank())         
tmax_extimates_plot

pdf('tmax_extimates_plot.pdf', height = 2, width = 1)
print(tmax_extimates_plot)
dev.off()
svg('tmax_extimates_plot.svg', height = 3.5, width = 3.5)
print(tmax_extimates_plot)
dev.off()




# phylogeny
#put that one species that we renamed in the dataframe back to the way it was before, and also just rename the tip in the phylogeny 
na.odes$tip.label <- revalue(na.odes$tip.label, c('Epigomphus_crepidus' = 'Epigomphus_quadracies'))
tol.clean$binom <- revalue(tol.clean$binom, c('Epigomphus_crepidus' = 'Epigomphus_quadracies'))
rownames(tol.clean) <- tol.clean$binom
plot(na.odes)


## we need to make a new little dataframe for the purposes of making this graph. graph the relevant columns from the datset. rename them something sensible
traits <- data.frame(tol.clean[,c("color","Tmax")])
rownames(traits) <- tol.clean$binom
colnames(traits) <- c('wing color', 'CTmax')

# make separate vectors for the colors 
tol.clean.ordered <- ReorderData(na.odes, tol.clean, taxa.names="row names")
wc <- ifelse(tol.clean.ordered$color == 'y', '#EBC06B', '#DBEAF3')
tc <- ifelse(tol.clean.ordered$color == 'y', '#714D21', '#83A2B1')

# this will help use know how wide to make the limits on the axis
range(tol.clean$Tmax) # 41.0 48.5

# this makes a special data object with both the phylogeny and the data in it. Necessary for making the joint graph
gphy4 <-phylo4d(na.odes, traits)
dotplot(gphy4, trait = 'CTmax', dot.col = wc, tip.labels = na.odes$tip.label, tip.col = tc, tip.cex = 1.1, center = FALSE, scale = FALSE, data.xlim = c(40.8, 48.8), tree.ratio = 0.4, grid.horizontal = FALSE, grid.vertical = FALSE, show.box = TRUE, trait.bg.col = 'white', dot.cex = 3)










#### Phylogenetic path analysis ####


# Define the model set with hypotheses. 
drag.mods <- define_model_set(
  cons.dir = c(color ~ Tmax),
  cons.dir.climT = c(color ~ Tmax, Tmax ~ climate),
  comp.dir = c(Tmax ~ color),
  comp.dir.climC = c(Tmax ~ color, color ~ climate)
)

# look at the models to make sure we have correctly specified the right pathways. 
plot_model_set(drag.mods)

# run the path analysis. Use pagel's lambda because it was best in PGLS
drag.result <- phylo_path(drag.mods, data = tol.clean, tree = na.odes, model = 'lambda')
drag.sum <- summary(drag.result)
drag.sum
#                        model k q     C     p CICc delta_CICc      l      w
# comp.dir             comp.dir 2 4 4.91 0.2965 15.8       0.00 1.000 0.5655
# cons.dir.climT cons.dir.climT 1 5 3.05 0.2174 17.7       1.90 0.387 0.2188
# comp.dir.climC comp.dir.climC 1 5 4.17 0.1244 18.8       3.01 0.221 0.1252
# cons.dir             cons.dir 2 4 8.58 0.0726 19.4       3.66 0.160 0.0905

drag.result$d_sep$comp.dir # lambda = 0.000000100

drag.avg <- average(drag.result, cut_off = 2)
drag.avg
#$coef
# climate             color       Tmax
# climate       0 0.0000000 -0.5172972
# color         0 0.0000000  0.9804985
# Tmax          0 0.9634248  0.0000000
# attr(,"class")
# [1] "matrix" "DAG"   
# 
# $se
# climate             color      Tmax
# climate       0 0.0000000 0.2602516
# color         0 0.0000000 0.4147988
# Tmax          0 0.5964054 0.0000000
# 
# $lower
# climate              color       Tmax
# climate       0  0.0000000 -1.0273810
# color         0  0.0000000  0.1675078
# Tmax          0 -0.2055083  0.0000000
# 
# $upper
# climate            color         Tmax
# climate       0 0.000000 -0.007213384
# color         0 0.000000  1.793489131
# Tmax          0 2.132358  0.000000000


# Best support for model where CTmax compensates for color evolution
#See which pathways in this model yield significant effects. Run a bootstrap with 500 iterations to determine confidence intervals for each path coefficient.
set.seed(1001)
drag.best <- best(drag.result, boot = 500)
#values
drag.best
#$coef
# climate         color      Tmax
# climate       0     0 0.0000000
# color         0     0 0.9804985
# Tmax          0     0 0.0000000
# 
# $se
# climate         color      Tmax
# climate       0     0 0.0000000
# color         0     0 0.4147988
# Tmax          0     0 0.0000000
# 
# $lower
# climate         color      Tmax
# climate       0     0 0.0000000
# color         0     0 0.1971374
# Tmax          0     0 0.0000000
# 
# $upper
# climate         color     Tmax
# climate       0     0 0.000000
# color         0     0 1.698185
# Tmax          0     0 0.000000


set.seed(1001)
drag.2best <- choice(drag.result, "cons.dir.climT", boot = 500)
drag.2best
#$coef
# climate               Tmax     color
# climate       0 -0.5172972 0.0000000
# Tmax          0  0.0000000 0.9634248
# color         0  0.0000000 0.0000000
# 
# $se
# climate              Tmax     color
# climate       0 0.2602516 0.0000000
# Tmax          0 0.0000000 0.5964054
# color         0 0.0000000 0.0000000
# 
# $lower
# climate              Tmax       color
# climate       0 -1.011246  0.00000000
# Tmax          0  0.000000 -0.00534753
# color         0  0.000000  0.00000000
# 
# $upper
# climate                 Tmax    color
# climate       0 -0.008656559 0.000000
# Tmax          0  0.000000000 2.604238
# color         0  0.000000000 0.000000

set.seed(1001)
drag.3best <- choice(drag.result, "comp.dir.climC", boot = 500)
drag.3best

set.seed(1001)
drag.4best <- choice(drag.result, "cons.dir", boot = 500)
drag.4best



#### climate plot ####

theme.drags <- theme(axis.text = element_text(color="Black"),
                     axis.title.x = element_text(size = 11), 
                     axis.text.x = element_text(size = 11),
                     axis.title.y = element_text(size = 11), 
                     axis.text.y = element_text(size = 9),
                     panel.border = element_blank(),
                     panel.background = element_blank(),
                     panel.grid.minor=element_blank(), panel.grid.major=element_blank(),
                     axis.line.x = element_line(linetype = 'solid', color = 'black', size = 0.5),
                     axis.line.y = element_line(linetype = 'solid', color = 'black', size = 0.5),
                     plot.margin = margin(1,1,5,1),
                     legend.key = element_rect(fill = 'white')) 


climate_plot <-
  ggplot() +  
  #geom_hline(yintercept = 0, size = 0.5, color = "grey", linetype = "dashed") +
  #geom_linerange(data = comp.diff.fit.dat, aes(x = trait, ymax = UCL, ymin = LCL, color = sig), size = 0.5, alpha = 1, ) + 
  geom_jitter(data = tol.clean, aes(x = climate, y= mean.bio1, pch = climate), alpha = 1, size=2, width = 0.2) + 
  scale_shape_manual(values = c(21, 16)) +
  scale_x_discrete(labels = c("Temperate\nspecies", "Tropical\nspecies")) +
  #scale_color_manual(values = c("grey", "black")) +
  scale_y_continuous(breaks = c(12,15,18,21,24,27), limits = c(11,27)) +
  #scale_y_continuous(breaks = c(-0.4,-0.2,0,0.2,0.4)) +
  ylab("Mean annual temperature across range (ºC)") +
  theme.drags +                                 
  theme(legend.position = "none", axis.title.x = element_blank())         
climate_plot

pdf('climate_plot.pdf', height = 3.5, width = 3.5)
print(climate_plot)
dev.off()
svg('climate_plot.svg', height = 3.5, width = 3.5)
print(climate_plot)
dev.off()





#### Supplementary analyses ####

#### PGLS May 1976 species only ####
# subset dataframe to only include tropicer species
tol.clean.May <- filter(tol.clean, May_1976 == "y")

# first we figure out which species are shared between the phylogeny and the subsetted dataframe
match_data_phylo.May <- match(na.odes$tip.label, tol.clean.May$binom)

# next we create a vector of names of species in the phylogeny that aren't in our dataframe
drop.species.May <- na.odes$tip.label[is.na(match_data_phylo.May)==TRUE]

# finally, we use the drop.tip() command to prune the tree so that includes only the species in our dataset
na.odes.May <- drop.tip(na.odes, drop.species.May)

# always a good idea to plot it
plot(na.odes.May)

# also a good idea to just look at the summary description of this phylogeny
summary(na.odes.May)


# now something that is kind of important for these kinds of analyses is that the phylogeny is "ultrametric", which essentially just means that the tips are all aligned. Sometimes you see phylogenies where the tips do not end in the same place, those are examples of phylogenies that are not ultrametric. From when we plotted the phylogeny, it sure looked like the tips were aligned, but it's always good to double check
is.ultrametric(na.odes.May) # TRUE


# do an AIC comparison of a brownian motion model, a model that estimates Pagel's lambda, and a model that doesn't include phylogeny.

tol00.brownian <- gls(Tmax_may ~ color, correlation = corPagel(1, phy = na.odes.May, fixed = TRUE), data = tol.clean.May, method = 'REML', na.action = na.fail)
tol00.pagels <- gls(Tmax_may ~ color, correlation = corPagel(1, phy = na.odes.May), data = tol.clean.May, method = 'REML', na.action =  na.fail)
tol00.pagels.5 <- gls(Tmax_may ~ color, correlation = corPagel(0.5, phy = na.odes.May), data = tol.clean.May, method = 'REML', na.action =  na.fail)
tol00.star <- gls(Tmax_may ~ color, correlation = corPagel(0, phy = na.odes.May, fixed = TRUE), data = tol.clean.May, method = 'REML', na.action =  na.fail)

# now let's see which of those models is the best fit to the data using AIC
model.sel(tol00.brownian, tol00.pagels, tol00.pagels.5, tol00.star, rank = "AIC")
#                (Intrc) color  correlation df  logLik  AIC delta weight
# tol00.pagels     44.52     +   c(1,n.o.M)  4 -18.713 45.4  0.00  0.402
# tol00.pagels.5   44.52     + c(0.5,n.o.M)  4 -18.713 45.4  0.00  0.402
# tol00.brownian   44.58     + c(1,n.o.M,T)  3 -20.578 47.2  1.73  0.169
# tol00.star       44.93     + c(0,n.o.M,T)  3 -22.434 50.9  5.44  0.026


# # Likelihood ratio tests pagels
tol01a <- gls(Tmax_may ~ color, correlation = corPagel(1, phy = na.odes.May), data = tol.clean.May, method = 'ML', na.action =  na.fail )
tol01b <- gls(Tmax_may ~ 1, correlation = corPagel(1, phy = na.odes.May), data = tol.clean.May, method = 'ML', na.action =  na.fail )
# test effect of color on tmax
anova(tol01a, tol01b)
#       Model df      AIC      BIC    logLik   Test  L.Ratio p-value
#tol01a     1  4 48.76538 51.02518 -20.38269                        
#tol01b     2  3 51.00007 52.69491 -22.50003 1 vs 2 4.234685  0.0396

summary(tol01a)
confint(tol01a)


# Likelihood ratio tests brownian
tol01a.brownian <- gls(Tmax_may ~ color, correlation = corPagel(1, phy = na.odes.May, fixed = TRUE), data = tol.clean.May, method = 'ML', na.action =  na.fail )
tol01b.brownian <- gls(Tmax_may ~ 1, correlation = corPagel(1, phy = na.odes.May, fixed = TRUE), data = tol.clean.May, method = 'ML', na.action =  na.fail )
# test effect of color on tmax
anova(tol01a.brownian, tol01b.brownian)
#                Model df      AIC      BIC    logLik   Test  L.Ratio p-value
#tol01a.brownian     1  3 51.34598 53.04083 -22.67299                        
#tol01b.brownian     2  2 52.00010 53.12999 -24.00005 1 vs 2 2.654113  0.1033

summary(tol01a.brownian)
confint(tol01a.brownian)

# Likelihood ratio tests star
tol01a.star <- gls(Tmax_may ~ color, correlation = corPagel(0, phy = na.odes.May, fixed = TRUE), data = tol.clean.May, method = 'ML', na.action =  na.fail )
tol01b.star <- gls(Tmax_may ~ 1, correlation = corPagel(0, phy = na.odes.May, fixed = TRUE), data = tol.clean.May, method = 'ML', na.action =  na.fail )
# test effect of color on tmax
anova(tol01a.star, tol01b.star)
#            Model df      AIC      BIC    logLik   Test  L.Ratio p-value
# tol01a.star     1  3 52.43686 54.13171 -23.21843                        
# tol01b.star     2  2 55.59765 56.72755 -25.79883 1 vs 2 5.160793  0.0231

summary(tol01a.star)
confint(tol01a.star)



#### PGLS Libellulidae only ####

# subset dataframe to only include Libellulidae
tol.clean.Libellulidae <- filter(tol.clean, family == "Libellulidae")

# first we figure out which species are shared between the phylogeny and the subsetted dataframe
match_data_phylo.Libellulidae <- match(na.odes$tip.label, tol.clean.Libellulidae$binom)

# next we create a vector of names of species in the phylogeny that aren't in our dataframe
drop.species.Libellulidae <- na.odes$tip.label[is.na(match_data_phylo.Libellulidae)==TRUE]

# finally, we use the drop.tip() command to prune the tree so that includes only the species in our dataset
na.odes.Libellulidae <- drop.tip(na.odes, drop.species.Libellulidae)

# always a good idea to plot it
plot(na.odes.Libellulidae)

# also a good idea to just look at the summary description of this phylogeny
summary(na.odes.Libellulidae)


# now something that is kind of important for these kinds of analyses is that the phylogeny is "ultrametric", which essentially just means that the tips are all aligned. Sometimes you see phylogenies where the tips do not end in the same place, those are examples of phylogenies that are not ultrametric. From when we plotted the phylogeny, it sure looked like the tips were aligned, but it's always good to double check
is.ultrametric(na.odes.Libellulidae) # TRUE

# do an AIC comparison of a brownian motion model, a model that estimates Pagel's lambda, and a model that doesn't include phylogeny at all. The way do to this is as such:
tol00.Libellulidae.brownian <- gls(Tmax ~ color, correlation = corPagel(1, phy = na.odes.Libellulidae, fixed = TRUE), data = tol.clean.Libellulidae, method = 'REML', na.action = na.fail)
tol00.Libellulidae.pagels <- gls(Tmax ~ color, correlation = corPagel(1, phy = na.odes.Libellulidae), data = tol.clean.Libellulidae, method = 'REML', na.action =  na.fail)
tol00.Libellulidae.pagels.5 <- gls(Tmax ~ color, correlation = corPagel(0.4, phy = na.odes.Libellulidae), data = tol.clean.Libellulidae, method = 'REML', na.action =  na.fail)
tol00.Libellulidae.pagels.0 <- gls(Tmax ~ color, correlation = corPagel(0, phy = na.odes.Libellulidae), data = tol.clean.Libellulidae, method = 'REML', na.action =  na.fail)
tol00.Libellulidae.star <- gls(Tmax ~ color, correlation = corPagel(0, phy = na.odes.Libellulidae, fixed = TRUE), data = tol.clean.Libellulidae, method = 'REML', na.action =  na.fail)

# now let's see which of those models is the best fit to the data using AIC
model.sel(tol00.Libellulidae.brownian, tol00.Libellulidae.pagels, tol00.Libellulidae.pagels.5, tol00.Libellulidae.pagels.0, tol00.Libellulidae.star, rank = "AIC")
#                             (Intrc) color  correlation df  logLik  AIC delta weight
# tol00.Libellulidae.star       44.02     + c(0,n.o.L,T)  3 -29.053 64.1  0.00  0.220
# tol00.Libellulidae.pagels.0   43.91     +   c(0,n.o.L)  4 -28.096 64.2  0.09  0.211
# tol00.Libellulidae.pagels     43.91     +   c(1,n.o.L)  4 -28.096 64.2  0.09  0.211
# tol00.Libellulidae.pagels.5   43.91     + c(0.4,n.o.L)  4 -28.096 64.2  0.09  0.211
# tol00.Libellulidae.brownian   44.10     + c(1,n.o.L,T)  3 -29.454 64.9  0.80  0.147
# pagel's converges

model.sel(tol00.Libellulidae.brownian, tol00.Libellulidae.pagels, tol00.Libellulidae.star, rank = "AIC")
#                             (Intrc) color  correlation df  logLik  AIC delta weight
# tol00.Libellulidae.star       44.02     + c(0,n.o.L,T)  3 -29.053 64.1  0.00  0.381
# tol00.Libellulidae.pagels     43.91     +   c(1,n.o.L)  4 -28.096 64.2  0.09  0.364
# tol00.Libellulidae.brownian   44.10     + c(1,n.o.L,T)  3 -29.454 64.9  0.80  0.255

# pagels
tol01.Libellulidaea.pagels <- gls(Tmax ~ color, correlation = corPagel(1, phy = na.odes.Libellulidae), data = tol.clean.Libellulidae, method = 'ML', na.action =  na.fail )
tol01.Libellulidaeb.pagels <- gls(Tmax ~ 1, correlation = corPagel(1, phy = na.odes.Libellulidae), data = tol.clean.Libellulidae, method = 'ML', na.action =  na.fail )
# test effect of color on tmax
anova(tol01.Libellulidaea.pagels, tol01.Libellulidaeb.pagels)
#                            Model df      AIC      BIC    logLik   Test  L.Ratio p-value
# tol01.Libellulidaea.pagels     1  4 68.19447 71.02667 -30.09724                        
# tol01.Libellulidaeb.pagels     2  3 63.04462 65.16878 -28.52231 1 vs 2 3.149849  0.0759
summary(tol01.Libellulidaea.pagels) #1.68360 0.9910373
confint(tol01.Libellulidaea.pagels) #-0.258793  3.626002

# brownian
tol01.Libellulidaea.brownian <- gls(Tmax ~ color, correlation = corPagel(1, phy = na.odes.Libellulidae, fixed = TRUE), data = tol.clean.Libellulidae, method = 'ML', na.action =  na.fail )
tol01.Libellulidaeb.brownian <- gls(Tmax ~ 1, correlation = corPagel(1, phy = na.odes.Libellulidae, fixed = TRUE), data = tol.clean.Libellulidae, method = 'ML', na.action =  na.fail )
# test effect of color on tmax
anova(tol01.Libellulidaea.brownian, tol01.Libellulidaeb.brownian)
#                              Model df      AIC      BIC    logLik   Test  L.Ratio p-value
# tol01.Libellulidaea.brownian     1  3 69.65021 71.77436 -31.82510                        
# tol01.Libellulidaeb.brownian     2  2 73.17670 74.59280 -34.58835 1 vs 2 5.526496  0.0187
summary(tol01.Libellulidaea.brownian) #1.75658 0.7299434
confint(tol01.Libellulidaea.brownian) #0.3259218  3.187247


# star
tol01.Libellulidaea.star <- gls(Tmax ~ color, correlation = corPagel(0, phy = na.odes.Libellulidae, fixed = TRUE), data = tol.clean.Libellulidae, method = 'ML', na.action =  na.fail )
tol01.Libellulidaeb.star <- gls(Tmax ~ 1, correlation = corPagel(0, phy = na.odes.Libellulidae, fixed = TRUE), data = tol.clean.Libellulidae, method = 'ML', na.action =  na.fail )
# test effect of color on tmax
anova(tol01.Libellulidaea.star, tol01.Libellulidaeb.star)
#                          Model df      AIC      BIC    logLik   Test  L.Ratio p-value
# tol01.Libellulidaea.star     1  3 66.25373 68.37788 -30.12687                        
# tol01.Libellulidaeb.star     2  2 68.05014 69.46624 -32.02507 1 vs 2 3.796411  0.0514
summary(tol01.Libellulidaea.star) #1.93964 1.0024218
confint(tol01.Libellulidaea.star) # -0.02506774  3.904353









#### PGLS color darkness ####

# do an AIC comparison of a brownian motion model, a model that estimates Pagel's lambda, and a model that doesn't include phylogeny.
tol.clean$color.darkness <- as.numeric(tol.clean$color.darkness)

tol00.brownian <- gls(log(color.darkness + 1) ~ Tmax, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'REML', na.action = na.fail)
tol00.pagels <- gls(log(color.darkness + 1) ~ Tmax, correlation = corPagel(1, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.pagels.5 <- gls(log(color.darkness + 1) ~ Tmax, correlation = corPagel(0.5, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.pagels.0 <- gls(log(color.darkness + 1) ~ Tmax, correlation = corPagel(0, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.star <- gls(log(color.darkness + 1) ~ Tmax, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'REML', na.action =  na.fail)

# now let's see which of those models is the best fit to the data using AIC
model.sel(tol00.brownian, tol00.pagels, tol00.pagels.5, tol00.pagels.0, tol00.star, rank = "AIC")
#                (Intrc)    Tmax   correlation df  logLik  AIC delta weight
# tol00.pagels    -3.247 0.09141   cP(1,na.od)  4  -6.365 20.7  0.00  0.257
# tol00.pagels.5  -3.247 0.09141 cP(0.5,na.od)  4  -6.365 20.7  0.00  0.257
# tol00.pagels.0  -3.247 0.09141   cP(0,na.od)  4  -6.365 20.7  0.00  0.257
# tol00.star      -2.594 0.07963 cP(0,na.od,T)  3  -7.474 20.9  0.22  0.230
# tol00.brownian  -4.128 0.10910 cP(1,na.od,T)  3 -13.544 33.1 12.36  0.001
# Pagel's converges

# # Likelihood ratio tests pagels
# problems with model if starting lambda value isn't specified, so starting at 0.3 since it works for both models
tol01a <- gls(log(color.darkness + 1) ~ Tmax, correlation = corPagel(0.3, phy = na.odes), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b <- gls(log(color.darkness + 1) ~ 1, correlation = corPagel(0.3, phy = na.odes), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.darkness on tmax
anova(tol01a, tol01b)
#       Model df       AIC       BIC    logLik   Test  L.Ratio p-value
# tol01a     1  4  13.32962  17.10738 -2.664811                        
# tol01b     2  3 -16.12945 -13.29613 11.064725 1 vs 2 27.45907  <.0001


# Likelihood ratio tests brownian
tol01a.brownian <- gls(log(color.darkness + 1) ~ Tmax, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b.brownian <- gls(log(color.darkness + 1) ~ 1, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.darkness on tmax
anova(tol01a.brownian, tol01b.brownian)
#                Model df      AIC      BIC    logLik   Test  L.Ratio p-value
# tol01a.brownian     1  3 29.26724 32.10056 -11.63362                        
# tol01b.brownian     2  2 39.32518 41.21405 -17.66259 1 vs 2 12.05794   5e-04


# Likelihood ratio tests star
tol01a.star <- gls(log(color.darkness + 1) ~ Tmax, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b.star <- gls(log(color.darkness + 1) ~ 1, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.darkness on tmax
anova(tol01a.star, tol01b.star)
#            Model df      AIC      BIC    logLik   Test  L.Ratio p-value
# tol01a.star     1  3 12.38736 15.22068 -3.193680                        
# tol01b.star     2  2 15.82416 17.71303 -5.912078 1 vs 2 5.436796  0.0197







# do an AIC comparison of a brownian motion model, a model that estimates Pagel's lambda, and a model that doesn't include phylogeny.
tol.clean$color.darkness <- as.factor(tol.clean$color.darkness)

tol00.brownian <- gls(Tmax ~ color.darkness, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'REML', na.action = na.fail)
tol00.pagels <- gls(Tmax ~ color.darkness, correlation = corPagel(1, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.pagels.5 <- gls(Tmax ~ color.darkness, correlation = corPagel(0.5, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.pagels.0 <- gls(Tmax ~ color.darkness, correlation = corPagel(0, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.star <- gls(Tmax ~ color.darkness, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'REML', na.action =  na.fail)

# now let's see which of those models is the best fit to the data using AIC
model.sel(tol00.brownian, tol00.pagels, tol00.pagels.5, tol00.pagels.0, tol00.star, rank = "AIC")
#                (Int) clr.drk   correlation df  logLik  AIC delta weight
# tol00.star     43.88       + cP(0,na.od,T)  4 -35.578 79.2  0.00  0.353
# tol00.pagels.5 44.28       + cP(0.5,na.od)  5 -35.075 80.1  0.99  0.215
# tol00.pagels.0 44.28       +   cP(0,na.od)  5 -35.075 80.1  0.99  0.215
# tol00.pagels   44.28       +   cP(1,na.od)  5 -35.075 80.1  0.99  0.215
# tol00.brownian 44.69       + cP(1,na.od,T)  4 -40.981 90.0 10.81  0.002
# Pagel's converges
# The star and Pagel's lambda model have the lowest AIC. Lets report results for all three to cover a range of phylogenetic signal values

# # Likelihood ratio tests pagels
# problems with model if starting lambda value is 1, so starting at 0.5
tol01a <- gls(Tmax ~ color.darkness, correlation = corPagel(0.5, phy = na.odes), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b <- gls(Tmax ~ 1, correlation = corPagel(0.5, phy = na.odes), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.darkness on tmax
anova(tol01a, tol01b)
#        Model df      AIC      BIC    logLik   Test  L.Ratio p-value
# tol01a     1  5 79.58629 84.30849 -34.79315                        
# tol01b     2  3 78.36176 81.19508 -36.18088 1 vs 2 2.775471  0.2496



# Likelihood ratio tests brownian
tol01a.brownian <- gls(Tmax ~ color.darkness, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b.brownian <- gls(Tmax ~ 1, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.darkness on tmax
anova(tol01a.brownian, tol01b.brownian)
#                 Model df       AIC      BIC    logLik   Test  L.Ratio p-value
# tol01a.brownian     1  4  99.65611 103.4339 -45.82806                        
# tol01b.brownian     2  2 109.17151 111.0604 -52.58575 1 vs 2 13.51539  0.0012


# Likelihood ratio tests star
tol01a.dark.star <- gls(Tmax ~ color.darkness, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b.dark.star <- gls(Tmax ~ 1, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.darkness on tmax
anova(tol01a.dark.star, tol01b.dark.star)
#             Model df      AIC      BIC    logLik   Test  L.Ratio p-value
# tol01a.star     1  4 84.07403 87.85179 -38.03701                        
# tol01b.star     2  2 85.54955 87.43843 -40.77478 1 vs 2 5.475523  0.0647

dark.cat.emmeans <- emmeans(tol01a.dark.star, "color.darkness")
dark.cat.emmeans
# color.darkness emmean    SE df lower.CL upper.CL
# 0                43.9 0.589 15     42.6     45.1
# 1                45.5 1.952 15     41.3     49.7
# 2                46.0 0.738 15     44.5     47.6



confint(emmeans(tol01a.dark.star, pairwise ~ "color.darkness", adjust = "none"))
# contrast                          estimate    SE df lower.CL upper.CL
#color.darkness0 - color.darkness1   -1.622 2.039 15    -5.97    2.724
#color.darkness0 - color.darkness2   -2.150 0.944 15    -4.16   -0.139
#color.darkness1 - color.darkness2   -0.529 2.087 15    -4.98    3.920




#### PGLS color position ####

# do an AIC comparison of a brownian motion model, a model that estimates Pagel's lambda, and a model that doesn't include phylogeny.
tol.clean$color.position <- as.numeric(tol.clean$color.position)
tol00.brownian <- gls(log(color.position + 1) ~ Tmax, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'REML', na.action = na.fail)
tol00.pagels <- gls(log(color.position + 1) ~ Tmax, correlation = corPagel(1, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.pagels.5 <- gls(log(color.position + 1) ~ Tmax, correlation = corPagel(0.5, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.pagels.0 <- gls(log(color.position + 1) ~ Tmax, correlation = corPagel(0, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.star <- gls(log(color.position + 1) ~ Tmax, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'REML', na.action =  na.fail)

# now let's see which of those models is the best fit to the data using AIC
model.sel(tol00.brownian, tol00.pagels, tol00.pagels.5, tol00.pagels.0, tol00.star, rank = "AIC")
#               (Intrc)    Tmax   correlation df logLik  AIC delta weight
# tol00.pagels.0  -3.939 0.10580   cP(0,na.od)  4 -1.869 11.7  0.00  0.303
# tol00.pagels    -3.939 0.10580   cP(1,na.od)  4 -1.869 11.7  0.00  0.303
# tol00.pagels.5  -3.939 0.10580 cP(0.5,na.od)  4 -1.869 11.7  0.00  0.303
# tol00.star      -3.180 0.09205 cP(0,na.od,T)  3 -4.145 14.3  2.55  0.084
# tol00.brownian  -4.590 0.11910 cP(1,na.od,T)  3 -6.570 19.1  7.40  0.007
# The star and Pagel's lambda model have the lowest AIC. Lets report results for all three to cover a range of phylogenetic signal values

# # Likelihood ratio tests pagels
# problems with model if starting lambda value isn't specified, so starting at 0.3 since it works for both models
tol01a <- gls(log(color.position + 1) ~ Tmax, correlation = corPagel(0.3, phy = na.odes), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b <- gls(log(color.position + 1) ~ 1, correlation = corPagel(0.3, phy = na.odes), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.position on tmax
anova(tol01a, tol01b)
#        Model df        AIC        BIC    logLik   Test  L.Ratio p-value
# tol01a     1  4   3.684978   7.462734  2.157511                        
# tol01b     2  3 -20.872603 -18.039286 13.436301 1 vs 2 22.55758  <.0001


# Likelihood ratio tests brownian
tol01a.brownian <- gls(log(color.position + 1) ~ Tmax, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b.brownian <- gls(log(color.position + 1) ~ 1, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.position on tmax
anova(tol01a.brownian, tol01b.brownian)
#                 Model df      AIC      BIC     logLik   Test L.Ratio p-value
# tol01a.brownian     1  3 13.67691 16.51022  -3.838453                       
# tol01b.brownian     2  2 34.93630 36.82518 -15.468152 1 vs 2 23.2594  <.0001


# Likelihood ratio tests star
tol01a.star <- gls(log(color.position + 1) ~ Tmax, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b.star <- gls(log(color.position + 1) ~ 1, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.position on tmax
anova(tol01a.star, tol01b.star)
#             Model df       AIC       BIC    logLik   Test  L.Ratio p-value
# tol01a.star     1  3  4.946074  7.779391  0.526963                        
# tol01b.star     2  2 12.517290 14.406168 -4.258645 1 vs 2 9.571216   0.002







# do an AIC comparison of a brownian motion model, a model that estimates Pagel's lambda, and a model that doesn't include phylogeny.
tol.clean$color.position <- as.factor(tol.clean$color.position)

tol00.brownian <- gls(Tmax ~ color.position, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'REML', na.action = na.fail)
tol00.pagels <- gls(Tmax ~ color.position, correlation = corPagel(1, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.pagels.5 <- gls(Tmax ~ color.position, correlation = corPagel(0.5, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)
tol00.pagels.0 <- gls(Tmax ~ color.position, correlation = corPagel(0, phy = na.odes), data = tol.clean, method = 'REML', na.action =  na.fail)

tol00.star <- gls(Tmax ~ color.position, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'REML', na.action =  na.fail)

# now let's see which of those models is the best fit to the data using AIC
model.sel(tol00.brownian, tol00.pagels, tol00.pagels.5, tol00.pagels.0, tol00.star, rank = "AIC")
#                (Int) clr.pst   correlation df  logLik  AIC delta weight
# tol00.pagels.0 44.41       +   cP(0,na.od)  5 -30.910 71.8  0.00  0.276
# tol00.pagels   44.41       +   cP(1,na.od)  5 -30.910 71.8  0.00  0.276
# tol00.pagels.5 44.41       + cP(0.5,na.od)  5 -30.910 71.8  0.00  0.276
# tol00.star     43.88       + cP(0,na.od,T)  4 -32.477 73.0  1.14  0.156
# tol00.brownian 44.65       + cP(1,na.od,T)  4 -34.736 77.5  5.65  0.016
# Pagel's converges
# The star and Pagel's lambda model have the lowest AIC. Lets report results for all three to cover a range of phylogenetic signal values

# # Likelihood ratio tests pagels
# problems with model if starting lambda value starts at 1, so starting at 0.5
tol01a <- gls(Tmax ~ color.position, correlation = corPagel(0.5, phy = na.odes), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b <- gls(Tmax ~ 1, correlation = corPagel(0.5, phy = na.odes), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.position on tmax
anova(tol01a, tol01b)
#        Model df      AIC      BIC    logLik   Test  L.Ratio p-value
# tol01a     1  5 75.69982 80.42201 -32.84991                        
# tol01b     2  3 78.36176 81.19508 -36.18088 1 vs 2 6.661947  0.0358




# Likelihood ratio tests brownian
tol01a.brownian <- gls(Tmax ~ color.position, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b.brownian <- gls(Tmax ~ 1, correlation = corPagel(1, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.position on tmax
anova(tol01a.brownian, tol01b.brownian)
#                 Model df       AIC       BIC    logLik   Test  L.Ratio p-value
# tol01a.brownian     1  4  83.19182  86.96957 -37.59591                        
# tol01b.brownian     2  2 109.17151 111.06038 -52.58575 1 vs 2 29.97969  <.0001



# Likelihood ratio tests star
tol01a.posit.star <- gls(Tmax ~ color.position, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
tol01b.posit.star <- gls(Tmax ~ 1, correlation = corPagel(0, phy = na.odes, fixed = TRUE), data = tol.clean, method = 'ML', na.action =  na.fail )
# test effect of color.position on tmax
anova(tol01a.star, tol01b.star)
#            Model df       AIC       BIC    logLik   Test  L.Ratio p-value
# tol01a.star     1  3  4.946074  7.779391  0.526963                        
# tol01b.star     2  2 12.517290 14.406168 -4.258645 1 vs 2 9.571216   0.002



position.cat.emmeans <- emmeans(tol01a.posit.star, "color.position")
position.cat.emmeans 
# color.position emmean    SE df lower.CL upper.CL
# 0                43.9 0.473 15     42.9     44.9
# 1                43.8 0.907 15     41.9     45.8
# 2                47.2 0.702 15     45.7     48.7

confint(emmeans(tol01a.posit.star, pairwise ~ "color.position", adjust = "none"))
# contrast                          estimate    SE df lower.CL upper.CL
# color.position0 - color.position1   0.0448 1.023 15    -2.14    2.225
# color.position0 - color.position2  -3.3618 0.847 15    -5.17   -1.556
# color.position1 - color.position2  -3.4067 1.147 15    -5.85   -0.962




